--- Kaliel's Tracker
--- Copyright (c) 2012-2019, Marouan Sabbagh <mar.sabbagh@gmail.com>
--- All Rights Reserved.
---
--- This file is part of addon Kaliel's Tracker.

local addonName, KT = ...
local M = KT:NewModule(addonName.."_Filters")
KT.Filters = M

local _DBG = function(...) if _DBG then _DBG("KT", ...) end end

-- Lua API
local ipairs = ipairs
local pairs = pairs
local strfind = string.find

-- WoW API
local _G = _G

local db, dbChar
local mediaPath = "Interface\\AddOns\\"..addonName.."\\Media\\"

local KTF = KT.frame
local OTF = ObjectiveTrackerFrame
local OTFHeader = OTF.HeaderMenu

local continents = KT.GetMapContinents()
local instanceQuestDifficulty = {
	[DIFFICULTY_DUNGEON_NORMAL] = { Enum.QuestTag.Dungeon },
	[DIFFICULTY_DUNGEON_HEROIC] = { Enum.QuestTag.Dungeon, Enum.QuestTag.Heroic },
	[DIFFICULTY_RAID10_NORMAL] = { Enum.QuestTag.Raid, Enum.QuestTag.Raid10 },
	[DIFFICULTY_RAID25_NORMAL] = { Enum.QuestTag.Raid, Enum.QuestTag.Raid25 },
	[DIFFICULTY_RAID10_HEROIC] = { Enum.QuestTag.Raid, Enum.QuestTag.Raid10 },
	[DIFFICULTY_RAID25_HEROIC] = { Enum.QuestTag.Raid, Enum.QuestTag.Raid25 },
	[DIFFICULTY_RAID_LFR] = { Enum.QuestTag.Raid },
	[DIFFICULTY_DUNGEON_CHALLENGE] = { Enum.QuestTag.Dungeon },
	[DIFFICULTY_RAID40] = { Enum.QuestTag.Raid },
	[DIFFICULTY_PRIMARYRAID_NORMAL] = { Enum.QuestTag.Raid },
	[DIFFICULTY_PRIMARYRAID_HEROIC] = { Enum.QuestTag.Raid },
	[DIFFICULTY_PRIMARYRAID_MYTHIC] = { Enum.QuestTag.Raid },
	[DIFFICULTY_PRIMARYRAID_LFR] = { Enum.QuestTag.Raid },
}
local factionColor = { ["Horde"] = "ff0000", ["Alliance"] = "007fff" }

local eventFrame

--------------
-- Internal --
--------------

local function SetHooks()
	local bck_ObjectiveTracker_OnEvent = OTF:GetScript("OnEvent")
	OTF:SetScript("OnEvent", function(self, event, ...)
		if event == "QUEST_ACCEPTED" then
			local _, questID = ...
			if db.filterAuto[1] then
				return
			end
		end
		bck_ObjectiveTracker_OnEvent(self, event, ...)
	end)

	-- Quests
	local bck_QuestObjectiveTracker_UntrackQuest = QuestObjectiveTracker_UntrackQuest
	QuestObjectiveTracker_UntrackQuest = function(dropDownButton, questID)
		if not db.filterAuto[1] then
			bck_QuestObjectiveTracker_UntrackQuest(dropDownButton, questID)
		end
	end
	
	local bck_QuestMapQuestOptions_TrackQuest = QuestMapQuestOptions_TrackQuest
	QuestMapQuestOptions_TrackQuest = function(questID)
		if not db.filterAuto[1] then
			bck_QuestMapQuestOptions_TrackQuest(questID)
		end
	end
end

local function IsInstanceQuest(questID)
	local _, _, difficulty, _ = GetInstanceInfo()
	local difficultyTags = instanceQuestDifficulty[difficulty]
	if difficultyTags then
		local tagID, tagName = GetQuestTagInfo(questID)
		for _, tag in ipairs(difficultyTags) do
			_DBG(difficulty.." ... "..tag, true)
			if tag == tagID then
				return true
			end
		end
	end
	return false
end

local function Filter_Quests(self, spec, idx)
	if not spec then return end
	local numEntries, _ = GetNumQuestLogEntries()

	KT.stopUpdate = true
	if KT_GetNumQuestWatches() > 0 then
		for i=1, numEntries do
			local _, _, _, isHeader, _, _, _, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) then
				KT_RemoveQuestWatch(questID)
			end
		end
	end

	if spec == "all" then
		for i=1, numEntries do
			local _, _, _, isHeader, _, _, _, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) then
				KT_AddQuestWatch(questID)
			end
		end
	elseif spec == "group" then
		for i=idx, numEntries do
			local _, _, _, isHeader, _, _, _, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) then
				KT_AddQuestWatch(questID)
			else
				break
			end
		end
		MSA_CloseDropDownMenus()
	elseif spec == "zone" then
		local zoneName = GetZoneText() or ""
		local subzoneName = GetSubZoneText() or ""
		local isInZone = false
		for i=1, numEntries do
			local title, _, _, isHeader, _, _, _, questID, _, _, isOnMap, _, isTask, isBounty = GetQuestLogTitle(i)
			if isHeader then
				isInZone = (title == zoneName or title == subzoneName)
			else
				if not isTask and (not isBounty or IsQuestComplete(questID)) and (isOnMap or isInZone) then
					if KT.inInstance then
						if IsInstanceQuest(questID) then
							KT_AddQuestWatch(questID)
						end
					else
						KT_AddQuestWatch(questID)
					end
				end
			end
		end
	elseif spec == "daily" then
		for i=1, numEntries do
			local _, _, _, isHeader, _, _, frequency, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) and frequency >= 2 then
				KT_AddQuestWatch(questID)
			end
		end
	elseif spec == "instance" then
		for i=1, numEntries do
			local _, _, _, isHeader, _, _, _, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) then
				local tagID, _ = GetQuestTagInfo(questID)
				if tagID == Enum.QuestTag.Dungeon or
					tagID == Enum.QuestTag.Heroic or
					tagID == Enum.QuestTag.Raid or
					tagID == Enum.QuestTag.Raid10 or
					tagID == Enum.QuestTag.Raid25 then
					KT_AddQuestWatch(questID)
				end
			end
		end
	elseif spec == "complete" then
		for i=1, numEntries do
			local _, _, _, isHeader, _, _, _, questID, _, _, _, _, isTask, isBounty = GetQuestLogTitle(i)
			if not isHeader and not isTask and (not isBounty or IsQuestComplete(questID)) and IsQuestComplete(questID) then
				KT_AddQuestWatch(questID)
			end
		end
	end
	KT.stopUpdate = false

	if dbChar.trackedQuests[1] then
		ObjectiveTracker_Update(OBJECTIVE_TRACKER_UPDATE_MODULE_QUEST, dbChar.trackedQuests[1].id)
	else
		ObjectiveTracker_Update(OBJECTIVE_TRACKER_UPDATE_MODULE_QUEST)
	end
	if QuestLogFrame:IsVisible() then
		QuestLog_Update()
	end
end

local DropDown_Initialize	-- function

local function DropDown_Toggle(level, button)
	local dropDown = KT.DropDown
	if dropDown.activeFrame ~= KTF.FilterButton then
		MSA_CloseDropDownMenus()
	end
	dropDown.activeFrame = KTF.FilterButton
	dropDown.initialize = DropDown_Initialize
	MSA_ToggleDropDownMenu(level or 1, button and MSA_DROPDOWNMENU_MENU_VALUE or nil, dropDown, KTF.FilterButton, -15, -1, nil, button or nil, MSA_DROPDOWNMENU_SHOW_TIME)
	if button then
		_G["MSA_DropDownList"..MSA_DROPDOWNMENU_MENU_LEVEL].showTimer = nil
	end
end

local function Filter_AutoTrack(self, id, spec)
	db.filterAuto[id] = (db.filterAuto[id] ~= spec) and spec or nil
	if db.filterAuto[id] then
		if id == 1 then
			Filter_Quests(self, spec)
		end
		KTF.FilterButton:GetNormalTexture():SetVertexColor(0, 1, 0)
	else
		if not (db.filterAuto[1] or db.filterAuto[2]) then
			KTF.FilterButton:GetNormalTexture():SetVertexColor(KT.hdrBtnColor.r, KT.hdrBtnColor.g, KT.hdrBtnColor.b)
		end
	end
	DropDown_Toggle()
end

local function GetInlineFactionIcon()
	local coords = QUEST_TAG_TCOORDS[strupper(KT.playerFaction)]
	return CreateTextureMarkup(QUEST_ICONS_FILE, QUEST_ICONS_FILE_WIDTH, QUEST_ICONS_FILE_HEIGHT, 22, 22
		, coords[1]
		, coords[2] - 0.02 -- Offset to stop bleeding from next image
		, coords[3]
		, coords[4])
end

function DropDown_Initialize(self, level)
	local numEntries, numQuests = GetNumQuestLogEntries()
	local info = MSA_DropDownMenu_CreateInfo()
	info.isNotRadio = true
	info.iconAtlas = false

	if level == 1 then
		info.notCheckable = true

		-- Quests
		info.text = TRACKER_HEADER_QUESTS
		info.isTitle = true
		MSA_DropDownMenu_AddButton(info)

		info.isTitle = false
		info.disabled = (db.filterAuto[1])
		info.func = Filter_Quests

		info.text = "All  ("..numQuests..")"
		info.hasArrow = not (db.filterAuto[1])
		info.value = 1
		info.arg1 = "all"
		MSA_DropDownMenu_AddButton(info)

		info.hasArrow = false

		info.text = "Zone"
		info.arg1 = "zone"
		MSA_DropDownMenu_AddButton(info)

		info.text = "Daily"
		info.icon = QUEST_ICONS_FILE
		info.tCoordLeft = QUEST_TAG_TCOORDS["DAILY"][1]
		info.tCoordRight = QUEST_TAG_TCOORDS["DAILY"][2]
		info.tCoordTop = QUEST_TAG_TCOORDS["DAILY"][3]
		info.tCoordBottom = QUEST_TAG_TCOORDS["DAILY"][4]
		info.arg1 = "daily"
		MSA_DropDownMenu_AddButton(info)

		info.text = "Instance"
		info.iconAtlas = true
		info.icon = "worldquest-icon-dungeon"
		info.arg1 = "instance"
		MSA_DropDownMenu_AddButton(info)

		info.iconAtlas = false

		info.text = "Completed"
		info.iconAtlas = false
		info.icon = QUEST_ICONS_FILE
		info.tCoordLeft = QUEST_TAG_TCOORDS["COMPLETED"][1]
		info.tCoordRight = QUEST_TAG_TCOORDS["COMPLETED"][2]
		info.tCoordTop = QUEST_TAG_TCOORDS["COMPLETED"][3]
		info.tCoordBottom = QUEST_TAG_TCOORDS["COMPLETED"][4]
		info.arg1 = "complete"
		MSA_DropDownMenu_AddButton(info)

		info.icon = nil

		info.text = "Untrack All"
		info.disabled = (db.filterAuto[1] or KT_GetNumQuestWatches() == 0)
		info.arg1 = ""
		MSA_DropDownMenu_AddButton(info)

		info.text = "|cff00ff00Auto|r Zone"
		info.notCheckable = false
		info.disabled = false
		info.arg1 = 1
		info.arg2 = "zone"
		info.checked = (db.filterAuto[info.arg1] == info.arg2)
		info.func = Filter_AutoTrack
		MSA_DropDownMenu_AddButton(info)
	elseif level == 2 then
		info.notCheckable = true

		if MSA_DROPDOWNMENU_MENU_VALUE == 1 then
			info.arg1 = "group"
			info.func = Filter_Quests

			if numEntries > 0 then
				local zoneName = GetZoneText() or ""
				local subzoneName = GetSubZoneText() or ""
				local headerTitle, headerOnMap, headerShown
				for i=1, numEntries do
					local title, _, _, isHeader, _, _, _, questID, _, _, isOnMap, _, isTask, isBounty, _, isHidden = GetQuestLogTitle(i)
					if isHeader then
						headerTitle = title
						headerOnMap = (isOnMap or title == zoneName or title == subzoneName)
						headerShown = false
					elseif not isTask and (not isBounty or IsQuestComplete(questID)) and not isHidden then
						if not headerShown then
							info.text = (headerOnMap and "|cff00ff00" or "")..headerTitle
							info.arg2 = i
							MSA_DropDownMenu_AddButton(info, level)
							headerShown = true
						end
					end
				end
			end
		end
	end
end

local function SetFrames()
	-- Event frame
	if not eventFrame then
		eventFrame = CreateFrame("Frame")
		eventFrame:SetScript("OnEvent", function(self, event, arg1, ...)
			_DBG("Event - "..event.." - "..(arg1 or ""), true)
			if event == "QUEST_ACCEPTED" then
				if db.filterAuto[1] == "zone" then
					KT.questStateStopUpdate = true
					Filter_Quests(_, "zone")
					KT.questStateStopUpdate = false
				end
			elseif event == "ZONE_CHANGED" or
					event == "ZONE_CHANGED_INDOORS" or
					event == "ZONE_CHANGED_NEW_AREA" then
				if db.filterAuto[1] == "zone" then
					Filter_Quests(_, "zone")
				end
			end
		end)
	end
	eventFrame:RegisterEvent("QUEST_ACCEPTED")
	eventFrame:RegisterEvent("ZONE_CHANGED")
	eventFrame:RegisterEvent("ZONE_CHANGED_INDOORS")
	eventFrame:RegisterEvent("ZONE_CHANGED_NEW_AREA")

	-- Filter button
	local button = CreateFrame("Button", addonName.."FilterButton", KTF)
	button:SetSize(16, 16)
	button:SetPoint("TOPRIGHT", KTF.MinimizeButton, "TOPLEFT", -4, 0)
	button:SetFrameLevel(KTF:GetFrameLevel() + 10)
	button:SetNormalTexture(mediaPath.."UI-KT-HeaderButtons")
	button:GetNormalTexture():SetTexCoord(0.5, 1, 0.5, 0.75)
	
	button:RegisterForClicks("AnyDown")
	button:SetScript("OnClick", function(self, btn)
		DropDown_Toggle()
	end)
	button:SetScript("OnEnter", function(self)
		self:GetNormalTexture():SetVertexColor(1, 1, 1)
		GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
		GameTooltip:AddLine("Filter", 1, 1, 1)
		GameTooltip:AddLine(db.filterAuto[1] and "- "..db.filterAuto[1].." Quests", 0, 1, 0)
		GameTooltip:Show()
	end)
	button:SetScript("OnLeave", function(self)
		if db.filterAuto[1] or db.filterAuto[2] then
			self:GetNormalTexture():SetVertexColor(0, 1, 0)
		else
			self:GetNormalTexture():SetVertexColor(KT.hdrBtnColor.r, KT.hdrBtnColor.g, KT.hdrBtnColor.b)
		end
		GameTooltip:Hide()
	end)
	KTF.FilterButton = button

	OTFHeader.Title:SetWidth(OTFHeader.Title:GetWidth() - 20)

	-- Move other buttons
	if db.hdrOtherButtons then
		local point, _, relativePoint, xOfs, yOfs = KTF.QuestLogButton:GetPoint()
		KTF.QuestLogButton:SetPoint(point, KTF.FilterButton, relativePoint, xOfs, yOfs)
	end
end

--------------
-- External --
--------------

function M:OnInitialize()
	_DBG("|cffffff00Init|r - "..self:GetName(), true)
	db = KT.db.profile
	dbChar = KT.db.char

    local defaults = KT:MergeTables({
        profile = {
            filterAuto = {
				nil,	-- [1] Quests
				nil,	-- [2] Achievements
			},
        }
    }, KT.db.defaults)
	KT.db:RegisterDefaults(defaults)
end

function M:OnEnable()
	_DBG("|cff00ff00Enable|r - "..self:GetName(), true)
	SetHooks()
	SetFrames()
end